/*
 *  Arnold emulator (c) Copyright, Kevin Thacker 1995-2015
 *
 *  This file is part of the Arnold emulator source code distribution.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
#ifndef __SNAPSHOT_V3_HEADER_INCLUDED__
#define __SNAPSHOT_V3_HEADER_INCLUDED__

#include "cpcglob.h"
#include "riff.h"
#include "snapshot.h"
#include <string>

/* handle a V3 chunk on reading */
void SnapshotV3_HandleChunk(const RIFF_CHUNK *pCurrentChunk,unsigned long Size, const SNAPSHOT_MEMORY_BLOCKS *pMemoryBlocks);
/* write the V3 CPC+ chunk */
unsigned char *SnapshotV3_CPCPlus_WriteChunk(unsigned char *buffer);

/* begin a chunk */
unsigned char *SnapshotV3_BeginChunk(unsigned char *buffer, unsigned long ChunkName);

/* write data to chunk */
unsigned char *SnapshotV3_WriteDataToChunk(unsigned char *buffer, const unsigned char *pData, unsigned long Length);

/* end a chunk */
void SnapshotV3_EndChunk(void);

unsigned long SnapshotV3_CPCPlus_CalculateOutputSize(void);

unsigned char *SnapshotV3_Memory_WriteChunk(unsigned char *buffer, const SNAPSHOT_MEMORY_BLOCKS *pMemoryBlocks, BOOL bCompressed);

BOOL Snapshot_DecompressBlock(const unsigned char *pChunkData, unsigned long nChunkLength, unsigned char *pOutputData, unsigned long nOutputLength);

enum
{
	SNAPSHOT_READ_CHUNK_OK,		// chunk is ok and we handled the data
	SNAPSHOT_READ_ERROR,		// chunk was incomplete
	SNAPSHOT_READ_CHUNK_HAS_SOME_UNSUPPORTED_DATA, // chunk read ok, but some data was unsupported.
};

class BlockReader
{
public:
	BlockReader(const unsigned char *pData, unsigned long Length)
	{
		m_pData = pData;
		m_LengthRemaining = Length;
	}

	bool ReadByte(unsigned char *pData)
	{
		if (m_LengthRemaining != 0)
		{
			*pData = m_pData[0];
			++m_pData;
			m_LengthRemaining--;
			return true;
		}
		return false;
	}
	bool ReadLongLE(unsigned long *pData)
	{
		unsigned char a, b, c, d;
		if (ReadByte(&a))
		{
			if (ReadByte(&b))
			{
				if (ReadByte(&c))
				{
					if (ReadByte(&d))
					{
						unsigned long Data = (a & 0x0ff) | ((b & 0x0ff) << 8) | ((c & 0x0ff) << 16) | ((d & 0x0ff) << 24);
						*pData = Data;
						return true;
					}
				}
			}
		}
		return false;
	}

	bool ReadLongBE(unsigned long *pData)
	{
		unsigned char a, b, c, d;
		if (ReadByte(&a))
		{
			if (ReadByte(&b))
			{
				if (ReadByte(&c))
				{
					if (ReadByte(&d))
					{
						unsigned long Data = ((a & 0x0ff)<<24) | ((b & 0x0ff) << 16) | ((c & 0x0ff) << 8) | (d & 0x0ff);
						*pData = Data;
						return true;
					}
				}
			}
		}
		return false;
	}
	bool ReadWordLE(unsigned short *pData)
	{
		unsigned char a, b;
		if (ReadByte(&a))
		{
			if (ReadByte(&b))
			{
				unsigned short Data = (a & 0x0ff) | ((b & 0x0ff) << 8);
				*pData = Data;
				return true;
			}
		}
		return false;
	}
	bool ReadWordBE(unsigned short *pData)
	{
		unsigned char a, b;
		if (ReadByte(&a))
		{
			if (ReadByte(&b))
			{
				unsigned short Data = ((a & 0x0ff)<<8) | (b & 0x0ff);
				*pData = Data;
				return true;
			}
		}
		return false;
	}
	bool ReadNullTerminatedString(std::string *pString)
	{
		if (!pString)
			return false;

		// record current position - note requires this position is accessible to setup string)
		const unsigned char *pStringStart = m_pData;

		// look for end of string
		unsigned char a;
		do
		{
			if (!ReadByte(&a))
				return false;

		} while (a != 0);

		int StringLength = m_pData - pStringStart;
		pString->assign(reinterpret_cast<const char *>(pStringStart), StringLength);
		
		return true;
	}

	bool ReadChunkAsString(std::string *pString)
	{
		if (!pString)
			return false;
		if (m_LengthRemaining == 0)
		{
			pString->empty();
			return true;
		}
		pString->assign(reinterpret_cast<const char *>(m_pData), m_LengthRemaining);
		m_LengthRemaining = 0;
		return true;
	}

	
	bool ReadLenAndString(std::string *pString)
	{
		if (!pString)
			return false;
		
		// read length
		unsigned char Length;
		if (!ReadByte(&Length))
			return false;

		// accept a string with no length
		if (Length == 0)
		{
			pString->empty();
			return true;
		}

		pString->reserve(Length);
		unsigned char ch;

		while (Length!=0)
		{ 
			if (!ReadByte(&ch))
				return false;

			pString->push_back(ch);
			Length--;
		}
		return true;
	}


	bool ReadLenAndStringWinape(std::string *pString)
	{
		if (!pString)
			return false;

		// read length
		unsigned long Length;
		if (!ReadLongLE(&Length))
			return false;

		// accept a string with no length
		if (Length == 0)
		{
			pString->empty();
			return true;
		}

		pString->reserve(Length);
		unsigned char ch;

		while (Length != 0)
		{
			if (!ReadByte(&ch))
				return false;

			pString->push_back(ch);
			Length--;
		}
		return true;
	}


	bool ReadBytes(unsigned char *pOutBuffer, int nBytes)
	{
		for (int i = 0; i < nBytes; i++)
		{
			if (!ReadByte(&pOutBuffer[i]))
				return false;
		}
		return true;
	}
	bool IsEOB() const
	{
		return (m_LengthRemaining == 0);
	}
private:
	const unsigned char *m_pData;
	unsigned long m_LengthRemaining;
};

typedef int (*SnapshotChunkReadHandler)(unsigned long ChunkID, BlockReader *pReader);
typedef int(*SnapshotChunkWriteHandler)(unsigned long ChunkID);
bool SnapshotV3_IsChunkSupported(unsigned long ChunkID);
void SnapshotV3_RegisterReadHandler(unsigned long ChunkID, SnapshotChunkReadHandler ReadHandler);
//void SnapshotV3_RegisterWriteHandler(unsigned long ChunkID, SnapshotChunkWriteHandler WriteHandler);


#endif

